<?php

/**
 * This is an example file to implement server-side caching capabilities
 * for the jQuery version of WeatherSlider. It can also be used without
 * caching to hide your World Weather Online API key from the source code,
 * so it will not be publicly visible to visitors (who might try to use it
 * for their own benefit).
 *
 * @package   WeatherSlider
 * @version   2.6.7
 * @author    Kreatura Media
 * @copyright Copyright (c) 2011-2016 Kreatura Media
 * @link      https://kreaturamedia.com/weatherslider-premium-jquery-weather-widget/
 * @license   https://codecanyon.net/licenses/terms/regular Envato Standard License
 */


/*
|--------------------------------------------------------------------------
| INSTALLATION INSTRUCTIONS
|--------------------------------------------------------------------------
|
| 1. Copy this file to the root (top-level) directory of your website.
|    - You can use an FTP app with the credentials provided by your host.
|    - In most cases you can also use online file managers usually provided
|      by web hosting companies.
|
|
| 2. Create a new folder called 'cache' (without the quotes) in the same directory.
|
|
| 3. Make sure that the newly created cache folder is writable.
|    - If you aren't familiar with file permissions, you should set CHMOD 755.
|
|
| 4. In your WeatherSlider sliders, you should add the 'ajaxURL' (without quotes)
|    property in the initialization code along with your other settings. It's an URL
|    pointing to this PHP file. In most cases it should be something like this:
|
|    $('#weatherslider').weatherSlider({
|        ajaxURL: '/weatherslider-cache-extension.php',
|        // Other options ....
|    })
|
|
| 5. Scroll down a bit in this document and review the settings below. There are
|    some required options and you will need to provide additional information
|    such as your WWO API key.
|
|
| 6. If you're experiencing issues, please triple check the settings below. It
|    will likely a misconfiguration issue. The usual suspects are the api_key,
|    api_version and api_premium settings.
|
|    Remember: Caching is enabled by default. Changes you make may not be visible
|              immediately without emptying the cache folder contents.
|
|
| 6. After successfully setting up the caching, you should no longer include the WWOAPIKey
|    option in your slider settings. This PHP script makes it possible to hide your key,
|    so it will not be publicly visible in the source code.
|
*/


$config = array(

	/*
	|--------------------------------------------------------------------------
	| Optional: Enable or disable caching
	|--------------------------------------------------------------------------
	|
	| This option controls the whether the script should use its caching
	| feature or retrieve data directly from WWO for each request.
	|
	| Use the value true to enable caching or false otherwise.
	|
	| Default: true (enabled caching)
	|
	*/

	'use_cache' => true,



	/*
	|--------------------------------------------------------------------------
	| Optional: Change the cache update frequency
	|--------------------------------------------------------------------------
	|
	| This option controls how frequently should the stored data be updated
	| from WWO. Set a positive number that represents the update frequency
	| in minutes. Greater values means rarer and fewer updates, thus it can lower
	| your API usage.
	|
	| Default: 30 (30 minutes)
	|
	*/

	'cache_interval' => 30,



	/*
	|--------------------------------------------------------------------------
	| REQUIRED: Specify the cache directory path
	|--------------------------------------------------------------------------
	|
	| Enter the filesystem path for the cache directory, relative to this file.
	| You don't have to change this setting if you follow our installation
	| instructions closely.
	|
	| Default: '/cache'
	|
	*/

	'cache_dir' => '/cache',



	/*
	|--------------------------------------------------------------------------
	| REQUIRED: Enter your World Weather Online API key
	|--------------------------------------------------------------------------
	|
	| Replace the <key> part below with your registered World Weather Online
	| API key. Make sure to not accidentally include any white spaces or extra
	| characters within the enclosing quotation marks.
	|
	| Default: '<key>' (no value)
	|
	*/

	'api_key' => '<key>',



	/*
	|--------------------------------------------------------------------------
	| REQUIRED: Specify the API version
	|--------------------------------------------------------------------------
	|
	| Free API keys registered before November 2014 and all Premium API keys
	| require setting the API version to be 1. Free API keys registered after
	| November 2014 require setting the API version to 2.
	|
	| As of April 5th, WWO no longer issue free API keys, thus newly created
	| accounts are considered as premium and you should set this value to 1.
	|
	| World Weather Online should tell you the API version assigned to your
	| account. Check your emails or log in to their website to make sure if
	| you're experiencing any issue.
	|
	| Default: 1
	|
	*/

	'api_version' => 1,



	/*
	|--------------------------------------------------------------------------
	| REQUIRED: Specify the API type
	|--------------------------------------------------------------------------
	|
	| Set whether you're using the free or premium API.
	|
	| As of April 5th, WWO no longer issue free API keys, thus newly created
	| accounts are considered as premium.
	|
	| Use the value true if you're using the premium API, set false otherwise.
	|
	| Default: true (Premium API)
	|
	*/

	'api_premium' => true,



	/*
	|--------------------------------------------------------------------------
	| OPTIONAL: Use SSL encryption
	|--------------------------------------------------------------------------
	|
	| Set whether you want to request data securely over the https:// protocol
	| with SSL encryption. In most cases you should not change the default.
	|
	| Default: true (use secure connection with encryption)
	|
	*/
	'api_ssl' => true,



	/*
	|--------------------------------------------------------------------------
	| DO NOT CHANGE: API URL
	|--------------------------------------------------------------------------
	|
	| This is the API URL used for requesting data from WWO.
	| You should not change this value, unless WWO introduce
	| currently unforeseeable API changes in the future.
	|
	| Default: 'api.worldweatheronline.com' (with qoutes)
	|
	*/
	'api_base' => 'api.worldweatheronline.com'

);











/*
|--------------------------------------------------------------------------
| DO NOT CHANGE THE REST OF THE FILE WITHOUT CODING EXPERINCE
|--------------------------------------------------------------------------
|
| This is the part when it gets serious. The below code should work just
| fine if you've set the options above correctly.
|
| IF YOU EXPERIENCE ANY ISSUE, PLEASE TRIPLE CHECK ALL THE SETTINGS ABOVE.
| MAKE SURE YOU'VE ENTERED THE CORRECT VALUES IN THE CORRECT SYNTAX.
|
| Do not modify the rest of this file without programming experience.
|
*/

define('ROOT_PATH', dirname( __FILE__ ) );

// Filter out invalid requests
if( empty($_GET['location']) || empty($_GET['service']) ) {
	exit;
}

// Get location & service type
$location 		= $_GET['location'];
$service 		= (isset($_GET['service']) && $_GET['service'] === 'timezone') ? 'timezone' : 'weather';

// Get the location of cache files
$file 			= ROOT_PATH.'/cache/'.base64_encode( $location.'_'.$service );
$file_weather 	= ROOT_PATH.'/cache/'.base64_encode( $location.'_weather' );
$file_timezone 	= ROOT_PATH.'/cache/'.base64_encode( $location.'_timezone' );


// Disabled caching:
// Retrieve the weather data directly from WWO,
// then stop the execution by outputting the data.
if( ! $config['use_cache'] ) {
	$data = fetch_data( $location );
	echo $data[ $service ];
	exit;
}



// Check whether a cache file already exists with
// up-to-date weather information.
if($data = get_cache( $file, $service )) {
	echo $data;
	exit;
}

// Otherwise continue execution and fetch fresh data from WWO
$data = fetch_data( $location );

// Save them
file_put_contents( $file_weather, $data['weather'] );
file_put_contents( $file_timezone, $data['timezone'] );

// Return it
echo $data[ $service ];
exit;






function remote_get( $url ) {

	$ch = curl_init();
	curl_setopt( $ch, CURLOPT_URL, $url );
	curl_setopt( $ch, CURLOPT_RETURNTRANSFER, 1 );
	curl_setopt( $ch, CURLOPT_USERAGENT, 'PHP/'.phpversion() );

	$output = curl_exec( $ch );
	curl_close( $ch );

	return $output;
}


function get_cache( $file, $service ) {

	global $config;

	// Cache exists and not expired
	if( file_exists( $file ) )  {
		if( filemtime( $file) > time() - $config['cache_interval'] * 60 ) {

			// Get file contents
			$content = file_get_contents( $file );

			// Local time fix
			if($service == 'timezone') {
				$content = json_decode( $content, true );
				$offset = $content['data']['time_zone'][0]['utcOffset'];

				$localtime = time() + ( 60 * 60 * $offset );
				$localtime = gmdate('Y-m-d H:i', $localtime );

				$content['data']['time_zone'][0]['localtime'] = $localtime;
				$content = json_encode( $content );
			}

			return $content;
		}
	}

	// Otherwise return false
	return false;
}


function fetch_data($location) {

	global $config;

	// Build URL for weather data
	$weatherURL = vsprintf('%s://%s/%s/v%s/weather.ashx?key=%s&format=json&q=%s&num_of_days=5', array(
		$config['api_ssl'] ? 'https' : 'http',
		$config['api_base'],
		$config['api_premium'] ? 'premium' : 'free',
		$config['api_version'],
		$config['api_key'],
		urlencode( $location ),
	));

	// Built URL for timezone data
	$timeZoneURL = vsprintf('%s://%s/%s/v%s/tz.ashx?key=%s&format=json&q=%s', array(
		$config['api_ssl'] ? 'https' : 'http',
		$config['api_base'],
		$config['api_premium'] ? 'premium' : 'free',
		$config['api_version'],
		$config['api_key'],
		urlencode( $location )
	));

	return array(
		'weather' 	=> remote_get( $weatherURL ),
		'timezone' 	=> remote_get( $timeZoneURL )
	);
}